//Ingenia Chat

const chatWindow = document.getElementById( "chat-window" );
const chatHeader = document.getElementById( "chat-header" );

let offsetX = 0, offsetY = 0, isDragging = false;

chatHeader.addEventListener( "mousedown", function ( e ) {
    isDragging = true;
    offsetX = e.clientX - chatWindow.offsetLeft;
    offsetY = e.clientY - chatWindow.offsetTop;
    document.body.style.userSelect = 'none';
} );

document.addEventListener( "mouseup", function () {
    isDragging = false;
    document.body.style.userSelect = 'auto';
} );

document.addEventListener( "mousemove", function ( e ) {
    if ( isDragging ) {
        chatWindow.style.left = `${e.clientX - offsetX}px`;
        chatWindow.style.top = `${e.clientY - offsetY}px`;
        chatWindow.style.bottom = 'auto';
        chatWindow.style.right = 'auto';
    }
} );

// Resizing
const resizer = document.getElementById( "chat-resizer" );
let isResizing = false;

resizer.addEventListener( "mousedown", function ( e ) {
    isResizing = true;
    e.preventDefault();
    document.body.style.userSelect = 'none';
} );

document.addEventListener( "mouseup", function () {
    isResizing = false;
    document.body.style.userSelect = 'auto';
} );

document.addEventListener( "mousemove", function ( e ) {
    if ( isResizing ) {
        const newWidth = e.clientX - chatWindow.offsetLeft;
        const newHeight = e.clientY - chatWindow.offsetTop;
        chatWindow.style.width = `${newWidth}px`;
        chatWindow.style.height = `${newHeight}px`;
    }
} );

document.getElementById( "chat-close" ).addEventListener( "click", function () {
    chatWindow.style.display = "none";
} );

// Input & Send Button
const inputField = document.getElementById( "chat-question" );
const sendButton = document.getElementById( "chat-send" );
const chatBody = document.getElementById( "chat-body" );

function extractChatMessages () {
    const chatBody = document.getElementById( "chat-body" );
    const messages = [];

    // Get all message elements
    const nodes = chatBody.querySelectorAll( ".message" );

    nodes.forEach( node => {
        // Skip typing indicator or other temporary elements
        if ( node.classList.contains( "typing-indicator" ) ) return;

        let role = "user";
        if ( node.classList.contains( "bot" ) ) {
            role = "assistant";
        }

        let text = node.cloneNode( true );

        messages.push( {
            role: role,
            content: text.textContent.trim()
        } );
    } );

    return messages;
}

function createBotMessage ( text ) {
    const container = document.createElement( "div" );
    container.className = "message bot";
    container.innerHTML = text;

    // const copyBtn = document.createElement( "button" );
    // copyBtn.className = "copy-btn";
    // copyBtn.title = "Copy to Clipboard";
    // copyBtn.textContent = "📋"; // Or use "Copy"

    // copyBtn.addEventListener( "click", ( e ) => {
    //     e.stopPropagation();
    //     navigator.clipboard.writeText( container.innerText ).then( () => {
    //         copyBtn.textContent = "✅";
    //         setTimeout( () => ( copyBtn.textContent = "📋" ), 1000 );
    //     } );
    // } );

    //container.appendChild( copyBtn );

    chatBody.appendChild( container );
    chatBody.scrollTop = chatBody.scrollHeight;
}

function showTypingIndicator () {
    const typing = document.createElement( "div" );
    typing.className = "message bot typing-indicator";
    typing.innerHTML = `<div class="typing"><em style="font-size: 0.8rem;">thinking</em><span>.</span><span>.</span><span>.</span></div>`;
    chatBody.appendChild( typing );
    chatBody.scrollTop = chatBody.scrollHeight;
    return typing;
}

function wrapMarkdownTablesInPre ( text ) {
    const lines = text.split( '\n' );
    const output = [];
    let currentTable = [];
    let inTable = false;

    for ( let i = 0; i < lines.length; i++ ) {
        const line = lines[ i ];

        const isTableLine = /^\s*\|.*\|\s*$/.test( line );
        const isSeparatorLine = /^\s*\|?\s*:?-+:?\s*(\|\s*:?-+:?\s*)+\|?\s*$/.test( line );

        if ( isTableLine || ( inTable && line.trim() === '' ) ) {
            currentTable.push( line );
            inTable = true;
        } else if ( inTable ) {
            // Output the pre block for table
            output.push( '<pre><code>' + currentTable.join( '\n' ) + '</code></pre>' );
            currentTable = [];
            inTable = false;
            output.push( line );
        } else {
            output.push( line );
        }
    }

    // Final table
    if ( currentTable.length > 0 ) {
        output.push( '<pre><code>' + currentTable.join( '\n' ) + '</code></pre>' );
    }

    return output.join( '\n' );
}

function formatCodeBlocksAndParagraphs ( text ) {
    const parts = [];
    const regex = /```(\w+)?\n([\s\S]*?)```/g;

    let lastIndex = 0;
    let match;

    var c = 0;
    while ( ( match = regex.exec( text ) ) !== null ) {
        const [ fullMatch, lang, code ] = match;
        const beforeCode = text.slice( lastIndex, match.index ).trim();

        if ( beforeCode ) {
            c = c + 1;
            // Convert plain text to paragraphs
            const paragraphs = beforeCode
                .split( /\n+/ )
                .map( p => `<p>${p.trim()}</p>` )
                .join( '\n' );
            parts.push( paragraphs );
        }

        // Convert code block
        parts.push(
            `<div title="Copy Code" class="codecopy" style="float: right; position: relative;" onclick="copyText(document.getElementById('c` + c + `'));">⧉</div><pre id="c` + c + `"><code>${escapeHtml( code.trim() )}</code></pre>`
        );

        lastIndex = match.index + fullMatch.length;
    }

    // Handle remaining text after last code block
    const remainingText = text.slice( lastIndex ).trim();
    if ( remainingText ) {
        const paragraphs = remainingText
            .split( /\n+/ )
            .map( p => `<p>${escapeHtml( p.trim() )}</p>` )
            .join( '\n' );
        parts.push( paragraphs );
    }

    return parts.join( '\n' );
}

function convertMarkdownLinksToAnchors ( text ) {
    const markdownLinkRegex = /\[([^\]]+)]\((https?:\/\/[^\s)]+)\)/g;
    return text.replace( markdownLinkRegex, '<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>' );
}

function formatCodeBlocks ( text ) {
    return text.replace( /```(\w+)?\n([\s\S]*?)```/g, ( match, lang, code ) => {
        return `<pre><code>${escapeHtml( code )}</code></pre>`;
    } );
}

// Optional: Escape HTML special characters inside code blocks
function escapeHtml ( str ) {
    return str
        .replace( /&/g, '&amp;' )
        .replace( /</g, '&lt;' )
        .replace( />/g, '&gt;' );
}

function sendMessage () {
    const msg = inputField.value.trim();
    if ( chatBody.querySelectorAll( ".typing-indicator" ).length != 0 ) {
        showError( "Ingenia is thinking" );
        return;
    }
    if ( msg ) {
        const msgElement = document.createElement( "div" );
        msgElement.className = "message user";
        msgElement.textContent = msg;
        chatBody.appendChild( msgElement );
        chatBody.scrollTop = chatBody.scrollHeight;
        inputField.value = "";

        let typingIndicator = showTypingIndicator();
        let _messages = extractChatMessages();
        var data = {
            id: document.getElementById( "chat-window" ).getAttribute( "chatid" ),
            messages: _messages,
            prompt: msg
        };
        ajax( "/api/ingeniallm", "interact", JSON.stringify( data ),
            ( success ) => {
                chatBody.removeChild( typingIndicator );
                try {
                    var j = JSON.parse( success );
                    var msg = JSON.parse( j.payload );
                    msg.response = msg.response.replace( /<think>[\s\S]*?<\/think>/g, '' );
                    createBotMessage( wrapMarkdownTablesInPre( convertMarkdownLinksToAnchors( formatCodeBlocksAndParagraphs( msg.response ) ) ) );
                } catch {
                    try {
                        var j = JSON.parse( success );
                        var msg = JSON.parse( j.payload );
                        createBotMessage( "Oh no, something went wrong: " + formatCodeBlocksAndParagraphs( msg.error ) );
                    } catch {
                        //normal message
                        var j = JSON.parse( success );
                        if ( j.message == undefined ) {
                            showSuccess( j.payload );
                            createBotMessage( "Task reported success: " + formatCodeBlocksAndParagraphs( j.payload.trim() ) );
                        } else {
                            showSuccess( j.payload );
                            createBotMessage( "Task reported error: " + formatCodeBlocksAndParagraphs( j.message.trim() ) );
                        }
                    }
                }
            },
            ( error ) => {
                chatBody.removeChild( typingIndicator );
                var j = JSON.parse( error );
                showError( j.message );
            }
        );
    }
}

document.getElementById( "clear-chat" ).addEventListener( "click", () => {
    const chatBody = document.getElementById( "chat-body" );
    chatBody.innerHTML = ""; // Clear all messages
    //Optionally scroll to top or show a placeholder
    chatBody.innerHTML = "<p><em><b>Welcome! I am Ingenia, what can I help you with?</b></em></p>";
} );

sendButton.addEventListener( "click", sendMessage );
inputField.addEventListener( "keydown", function ( e ) {
    if ( e.key === "Enter" && !e.shiftKey ) {
        e.preventDefault(); // prevent newline
        sendMessage(); // submit message
    }
    // Ctrl+Enter or Shift+Enter will insert a newline automatically
} );